<?php
namespace WPSURL\App;
/**
 * Exit if accessed directly
 */
if (!defined('ABSPATH')) {
    exit;
}

class Utility {
    /**
    *  preg replace slug string by regex
    *
    * @param $slug
    *
    * @return false|string|string[]|null
    */
    public static function slugRegexMatch($slug){
       $slug = preg_replace('~[^\pL\d]+~u', '-', $slug);
       $slug = iconv('utf-8', 'us-ascii//TRANSLIT', $slug);
       $slug = preg_replace('~[^-\w]+~', '', $slug);
       $slug = trim($slug, '-');
       $slug = preg_replace('~-+~', '-', $slug);
       $slug = strtolower($slug);
       if (empty($slug)) {
           return 'n-a';
       }
   
       return $slug;
   }

    /**
     *
     * Get http response code
     *
     * @param $theURL
     *
     * @return bool|string
     */
    public static function getHttpResponseCode($theURL)
    {
        $headers = get_headers($theURL);

        return isset($headers[0]) ? substr($headers[0], 9, 3) : false;
    }

    /**
     * Clean and Alternative Specials Char in String
     *
     * @param $str
     * @param string $alternate
     *
     * @return string|string[]|null
     */
    public static function cleanSpecialChar($str, $alternate = '')
    {
        $str = str_replace(' ', '_', $str);

        return preg_replace('/[^A-Za-z0-9\-]/', $alternate, $str);
    }

    /**
     *
     * check is exists wp rewrite rules
     *
     * @param $regex
     *
     * @return bool
     */
    public static function isExistsRewrite($regex)
    {
        $rules = \WPSURL\App\Options::getOption('rewrite_rules');
        if (!empty($regex)) {
            if (isset($rules[$regex])) {
                return true;
            }
        }

        return false;
    }

    /**
     *
     * Get visitor ip
     *
     * @return mixed
     */
    public static function getVisitorIP()
    {
        if (isset($_SERVER["HTTP_CF_CONNECTING_IP"])) {
            $_SERVER['REMOTE_ADDR'] = $_SERVER["HTTP_CF_CONNECTING_IP"];
            $_SERVER['HTTP_CLIENT_IP'] = $_SERVER["HTTP_CF_CONNECTING_IP"];
        }
        $client = isset($_SERVER['HTTP_CLIENT_IP']) ? $_SERVER['HTTP_CLIENT_IP'] : null;
        $forward = isset( $_SERVER['HTTP_X_FORWARDED_FOR']) ?  $_SERVER['HTTP_X_FORWARDED_FOR'] : null;
        $remote = $_SERVER['REMOTE_ADDR'];
        if (filter_var($client, FILTER_VALIDATE_IP)) {
            $ip = $client;
        } elseif (filter_var($forward, FILTER_VALIDATE_IP)) {
            $ip = $forward;
        } else {
            $ip = $remote;
        }

        return $ip;
    }

    /**
     *  Get visitor geo info by api 'geoplugin.net'
     *
     * @param string $name
     *
     * @return string|null
     */
    public static function getVisitorGeoInfo($name = 'cn', $ip = null)
    {
        global $geoplugin;
        if (empty($ip)) {
            $ip = self::getVisitorIP();
        }
        $geoplugin->locate($ip);
        /**
         *
         *  Returned data by api
         *
         *  cn = countryName
         *  ccode = countryCode
         *  c = city
         *  ctn = continentName
         *  lt = latitude
         *  lot = longitude
         *  cs = currencySymbol
         *  cc = currencyCode
         *  tz = timezone
         *
         */
        $data = array(
            'CN' => $geoplugin->countryName,
            'CCODE' => $geoplugin->countryCode,
            'C' => $geoplugin->city,
            'CTN' => $geoplugin->continentName,
            'LT' => $geoplugin->latitude,
            'LOT' => $geoplugin->longitude,
            'CS' => $geoplugin->currencySymbol,
            'CC' => $geoplugin->currencyCode,
            'TZ' => $geoplugin->timezone
        );
        $res = $data[strtoupper($name)];
        if (!empty($res)) {
            return $res;
        }

        return null;
    }

    /**
     *
     * show html country flag by  client ip
     *
     * @param null $ip
     */
    public static function showCountryFlagHtml($ip = null)
    {
        $countryCode = strtolower(self::getVisitorGeoInfo('ccode', $ip));
        $countryName = strtolower(self::getVisitorGeoInfo('cn', $ip));
        $flag_api = 'https://ipdata.co/flags/' . $countryCode . '.png';
        if (!empty($countryCode)) {
            echo '<img src="' . $flag_api . '" title="' . $countryName . '" alt="' . $countryName . '" />';
        } else {
            echo '<span title="' . __('Unknown', WPSURL_TEXT_DOMAIN) . '"><i class="wpsurl-icon-question-circle"></i></span>';
        }
    }

    /**
     *
     * get wp safe url view page slug
     *
     * @param int $type
     *
     * @return bool|mixed|string|void
     */
    public static function getPagesSlug(int $type = 1)
    {
        $slug = Settings::getOption('redirect_page_slug');
        $short_slug = Settings::getOption('redirect_page_short_slug');
        if ($type === 1) {
            if (!empty($slug)) {
                return $slug;
            } else {
                return 'safe-url';
            }
        } elseif ($type === 2) {
            if (!empty($short_slug)) {
                return $short_slug;
            } else {
                return 'surl';
            }
        } elseif($type === 3){
            if (!empty($short_slug)) {
                return $short_slug . '-download';
            } else {
                return 'surl-download';
            }
        }

        return false;
    }

    /**
     *
     * get wp safe url view current page url
     *
     * @return string|void
     */
    public static function getCurrentViewPageUrl()
    {
        if (self::issetQueryVars(self::getPagesSlug())) {
            return site_url('/' . self::getPagesSlug() . '/' . self::getQueryVarsValue());
        }
        return site_url('/' . self::getPagesSlug(2) . '/' . self::getQueryVarsValue());
    }

    /**
     *
     * get wp safe url view current page title
     *
     * @return bool
     */
    public static function getCurrentViewPageTitle()
    {
        if (self::issetQueryVars(self::getPagesSlug(2))) {
            return Links::getItem(self::getQueryVarsValue(), 'title');
        } else {
            return Links::getItem(self::getQueryVarsValue(), 'title');
        }
    }

    /**
     *
     * generate random id
     *
     * @param int $len
     *
     * @return string
     */
    public static function generateRandomID($len = 6)
    {
        $char = '123456789';
        $charLen = strlen($char);
        $id = '';
        for ($i = 0; $i < $len; $i++) {
            $id .= $char[rand(0, $charLen - 1)];
        }

        return $id;
    }

    /**
     *
     * generate unique link id
     *
     * @return bool|string|integer
     */
    public static function generateUniqueLinkID()
    {
        $id = self::generateRandomID();
        do {
            if (!Links::getByID($id)) {
                return $id;
            }
            $id = self::generateRandomID();
        } while (true);
    }

    public static function generateUniqueLinkKey()
    {
        return md5(time());
    }

    /**
     *
     * Get svg file content
     *
     * @param $file
     *
     * @return false|string
     */
    public static function getInlineSvg($file)
    {
        $file_type = pathinfo($file, PATHINFO_EXTENSION);
        if (strtolower($file_type) == 'svg') {
            return file_get_contents($file);
        }

        return $file;
    }

    /**
     * show html loader
     *
     * @param string $bg
     */
    public static function showLoader($bg = '#ffffff', $type = 'html', $svg_mode = 'light')
    {
        if ($type == 'html') {
            $c = $bg !== '#ffffff' ? ' style="background:' . $bg . ';"' : '';
            echo '<div class="wpsurl-loader"><div' . $c . '></div><div' . $c . '></div><div' . $c . '></div><div' . $c . '></div></div>';
        } elseif ($type = 'svg') {
            $file = WPSURL_IMG_URL . 'loader.svg';
            echo '<div class="wpsurl-svg-loader ' . $svg_mode . '">' . self::getInlineSvg($file) . '</div>';
        }
    }

    /**
     *
     * show html message
     *
     * @param $msg
     * @param string $type
     */
    public static function showHtmlMsg($msg, $type = 'success', $exit = false)
    {
        $cls = WPSURL_IS_ADMIN ? $type . ' settings-error notice is-dismissible' : $type;
        if(Utility::isExistsQueryVarValue()){
            $cls .= ' is-link-page-alert';
        }
        echo '<div class="wpsurl-alert ' . $cls . '"><p>' . $msg . '</p></div>';
        if ($exit) {
            exit();
        }
    }

    /**
     *
     * Convert links to full hash URLs
     *
     * @param $token
     * @param int $type
     *
     * @return string|boolean
     */
    public static function getHashLink($token, $type = 1)
    {
        $url = site_url('/' . self::getPagesSlug(($type === 1 ? 1 : ($type === 3 ? 3 : 2))) . '/{{KEY}}');
        if (!empty($token)) {
            if(is_int(intval($token))){
                $link = Links::get($token);
                if($link){
                    return str_replace('{{KEY}}', ($type === 1 ? $link->key : $link->unique_id), $url);
                }
            } else {
                return str_replace('{{KEY}}', ($type === 1 ? md5($token) : $token), $url);
            }
        }

        return false;
    }

    /**
     *
     * get short link by id
     *
     * @param $unique_id
     *
     * @return string|void
     */
    public static function getShortHashLink($unique_id)
    {
        return !empty($unique_id) && Links::getByID($unique_id) ? site_url('/' . self::getPagesSlug(2) . '/' . $unique_id) : $unique_id;
    }

    /**
     * @param $unique_id
     * @return mixed
     */
    public static function getDownloadLink($unique_id){
        $password = Links::getItem($unique_id, 'password');

        if (!empty($password)) {
            return site_url(\WPSURL\App\Utility::getPagesSlug(2) . '-download/' . $unique_id . '/?ac=' . base64_encode($password));
        }

        return site_url(\WPSURL\App\Utility::getPagesSlug(2) . '-download/' . $unique_id);
    }

    /*
    *
    *  Get page number
    *
    */
    public static function getCurrentPageNum()
    {
        return isset($_GET['page_num']) ? abs((int)$_GET['page_num']) : 1;
    }

    public static function cleanPageNumQueryVar()
    {
        $result = remove_query_arg('page_num');

        return add_query_arg(array('page_num' => '%#%'), $result);
    }

    public static function renderPagination($per_page = 15, $total = 0)
    {
        $page = self::getCurrentPageNum();
        $all = $total;
        $max_num_pages = ceil($all / $per_page);
        if ($max_num_pages > 1) {
            echo '<span class="button" style="margin:0 20px">' . __('Page', WPSURL_TEXT_DOMAIN) . ' ' . $page . ' ' . __('Of', WPSURL_TEXT_DOMAIN) . ' ' . $max_num_pages . '</span>';
            echo paginate_links(array(
                'base' => self::cleanPageNumQueryVar(),
                'format' => '&page_num=%#%',
                'total' => $max_num_pages,
                'current' => $page,
                'prev_text' => __('Prev', WPSURL_TEXT_DOMAIN),
                'next_text' => __('Next', WPSURL_TEXT_DOMAIN)
            ));
        }
    }

    /**
     *
     * check isset view page query vars
     *
     * @param null $query
     *
     * @return bool
     */
    public static function issetQueryVars($query = null)
    {
        global $wp;
        if (empty($query) || $query == null || $query == '' && !isset($wp->query_vars[$query])) {
            if (isset($wp->query_vars[self::getPagesSlug()])) {
                return true;
            } elseif (isset($wp->query_vars[self::getPagesSlug(2)])) {
                return true;
            }
        } else {
            if (isset($wp->query_vars[$query])) {
                return true;
            }
        }

        return false;
    }

    /**
     *
     * get view pages query vars value
     *
     * @param null $query
     *
     * @return string|boolean
     */
    public static function getQueryVarsValue($query = null)
    {
        global $wp;
        if (empty($query) && !isset($wp->query_vars[$query])) {
            if (isset($wp->query_vars[self::getPagesSlug()])) {
                return $wp->query_vars[self::getPagesSlug()];
            } elseif (isset($wp->query_vars[self::getPagesSlug(2)])) {
                return $wp->query_vars[self::getPagesSlug(2)];
            } elseif (isset($wp->query_vars[self::getPagesSlug(3)])) {
                return $wp->query_vars[self::getPagesSlug(3)];
            }
        } else {
            if (isset($wp->query_vars[$query])) {
                return $wp->query_vars[$query];
            }
        }

        return false;
    }

    /**
     *
     * check is exists view page query var value
     *
     * @return bool
     */
    public static function isExistsQueryVarValue()
    {
        global $wp;
        if (isset($wp->query_vars[self::getPagesSlug()]) && Links::getByKey($wp->query_vars[self::getPagesSlug()])) {
            return true;
        } elseif (isset($wp->query_vars[self::getPagesSlug(2)]) && Links::getByID($wp->query_vars[self::getPagesSlug(2)])) {
            return true;
        }

        return false;
    }

    public static function isValidQueryVarToken($token)
    {
        return self::issetQueryVars( self::getPagesSlug() ) && ! preg_match( '/^[a-zA-Z0-9]*$/', (string) $token ) || self::issetQueryVars( self::getPagesSlug( 2 ) ) && ! preg_match( '/^[1-9][0-9]*$/', (string) $token );
    }

    /**
     *
     * check equal password
     *
     * @param $pwd1
     * @param $pwd2
     *
     * @return bool
     */
    public static function isEqualPassword($pwd1, $pwd2)
    {
        if (!empty($pwd1) && !empty($pwd2)) {
            if ($pwd1 === $pwd2) {
                return true;
            }
        }

        return false;
    }

    /**
     *
     * html tags un slash
     *
     * @param $value
     *
     * @return string
     */
    public static function unslashHtml($value)
    {
        if (!empty($value)) {
            return html_entity_decode(htmlspecialchars_decode(wp_unslash($value)));
        }

         return false;
    }

    public static function hasNavMenu($location)
    {
        return (in_array($location, array_flip(get_nav_menu_locations()))) ? true : false;
    }

    public static function dropDownPosts($args = '')
    {
        $defaults = array(
            'post_type' => 'post',
            'selected' => null,
            'pagination' => false,
            'posts_per_page' => -1,
            'post_status' => 'publish',
            'cache_results' => true,
            'cache_post_meta_cache' => true,
            'echo' => true,
            'select_name' => 'post_id',
            'id' => '',
            'class' => '',
            'show' => 'post_title',
            'show_callback' => null,
            'show_option_all' => null,
            'show_option_none' => null,
            'option_none_value' => '',
            'multi' => false,
            'value_field' => 'ID',
            'order' => 'ASC',
            'orderby' => 'post_title',
        );
        $r = wp_parse_args($args, $defaults);
        $posts = get_posts($r);
        $output = '';
        $show = $r['show'];
        if (!empty($posts)) {

            $name = esc_attr($r['select_name']);
            if ($r['multi'] && !$r['id']) {
                $id = '';
            } else {
                $id = $r['id'] ? " id='" . esc_attr($r['id']) . "'" : " id='$name'";
            }
            $multiply = ($r['multi']) ? " multiple" : "";
            $output = "<select name='{$name}'{$id} class='" . esc_attr($r['class']) . "'" . $multiply . ">\n";
            if ($r['show_option_all']) {
                $output .= '<option value="0">' . $r['show_option_all'] . '</option>' . PHP_EOL;
            }
            if ($r['show_option_none']) {
                $_selected = selected($r['show_option_none'], $r['selected'], false);
                $output .= "<option value='" . esc_attr($r['option_none_value']) . "'$_selected>{$r['show_option_none']}</option>\n";
            }
            foreach ((array)$posts as $post) {
                $post->ID = (int)$post->ID;
                if (is_array($r['selected'])) {
                    $_selected = (in_array($post->ID, $r['selected'])) ? 'selected' : '';
                } else {
                    $_selected = selected($post->ID, $r['selected'], false);
                }
                $value = !isset($r['value_field']) || !isset($post->{$r['value_field']}) ? $post->ID : $post->{$r['value_field']};
                $display = !empty($post->$show) ? $post->$show : sprintf(__('#%d (no title)'), $post->ID);
                if ($r['show_callback']) {
                    $display = call_user_func($r['show_callback'], $display, $post->ID);
                }
                $output .= '<option value="' . $value . '"' . $_selected . '>' . esc_html($display) . '</option>';
            }
            $output .= "</select>";
        }
        /**
         * Filter the HTML output of a list of pages as a drop down.
         *
         * @param string $output HTML output for drop down list of pages.
         * @param array $r The parsed arguments array.
         * @param array $posts List of WP_Post objects returned by `get_pages()`
         *
         * @since 4.4.0 `$r` and `$posts` added as arguments.
         *
         * @since 2.1.0
         */
        $html = apply_filters('wp_dropdown_posts', $output, $r, $posts);
        if ($r['echo']) {
            echo $html;
        }

        return $html;
    }

    /**
     *
     * Get content translation id
     *
     * Compatible: WPML / PolyLang
     *
     */
    public static function getContentTranslationID($content_id){
        $mth = preg_match('/[a-z_A-Z]{0,5}.(?=_)/', get_locale(), $matches);
        $lang = isset($matches[0]) && !empty($matches[0]) ? strtolower($matches[0]) : 'fa';
        $post_type = get_post_type($content_id);
        $translate_id = 0;
        if(defined('ICL_PLUGIN_PATH')){
            $translation_id = apply_filters('wpml_object_id', $content_id, $post_type, FALSE, $lang);
            $translate_id = intval($translation_id) ? $translation_id : 0;
        } elseif(function_exists('pll_get_post_translations')){
            $lang = pll_current_language();
            $translations = pll_get_post_translations($content_id);
            $translate_id = (isset($translations[$lang]) && intval($translations[$lang])) ? $translations[$lang] : 0;
        }
        $content_id = intval($translate_id) ? $translate_id : $content_id;
        return apply_filters('wpsurl_get_content_translation_id', $content_id);
    }

    /**
     *
     * Json validation and return
     *
     * @param $str
     * @param $assoc
     * @return false|mixed
     */
    public static function isJson($str, $assoc = false)
    {
        $str = json_decode($str, $assoc);
        return $str && json_last_error() === JSON_ERROR_NONE ? $str : false;
    }

    /**
     *
     * Generate security nonce
     *
     * @return mixed
     */
    public static function generateNonce()
    {
        return wp_create_nonce('wpsurl_nonce');
    }

    /**
     *
     * Convert time to remaining days
     *
     * @param $time
     * @return false|float
     */
    public static function convertToRemainingDays($time)
    {
        $current_time = time();
        $remaining_seconds = $time - $current_time;

        return floor($remaining_seconds / (60 * 60 * 24));
    }

    public static function convertToLeftDays($time)
    {
        $current_time = time();
        $left_seconds = $current_time - $time;

        return floor($left_seconds / (60 * 60 * 24));
    }

    public static function date($format, $timestamp = null, $timezone = null){
        $date = wp_date($format, $timestamp, $timezone);
        $date = apply_filters('wpsurl_wp_date', $date);
        return $date;
    }

    public static function getTimeZone()
    {
        $timezone = 'UTC';
        $timezone = apply_filters('wpsurl_get_timezone', $timezone);
        return new \DateTimeZone($timezone);
    }
}