<?php
namespace WPSURL\App\Api;

use WPSURL\App\LinkOptions;
use WPSURL\App\Links;
use WPSURL\App\Options;
use WPSURL\App\Utility;

class PluginApiRoutes extends RestApi
{
    /**
     * @registerEndpoint register custom endpoint
     *
     * @return void
     */
    public function registerEndpoint($rest)
    {
        register_rest_route('surl', '/create', array(
            'methods'  => 'POST',
            'show_in_index' => false,
            'callback' => [$this, 'handleCreateResponse'],
        ));

        register_rest_route('surl', '/get', array(
            'methods'  => 'GET',
            'show_in_index' => false,
            'callback' => [$this, 'handleGetResponse'],
        ));

        register_rest_route('surl', '/delete', array(
            'methods'  => 'DELETE',
            'show_in_index' => false,
            'callback' => [$this, 'handleRemoveResponse'],
        ));
    }

    /**
     *
     * @param $request
     */
    public function handleCreateResponse($request)
    {
        $this->request = $request;
        $method = $this->request->get_method();
        $body = $this->getBody();

        if($method !== 'POST'){
            return self::parseError('Method Not Allowed!', 405);
        }

        $many_request = $this->parseManyRequest();
        if(is_wp_error($many_request)){
            return $many_request;
        }

        $auth = $this->authorization($this->request, 'write');
        if(is_wp_error($auth)){
            return $auth;
        }

        $bad_request = $this->parseBadRequest($body, ['title', 'type']);
        if(is_wp_error($bad_request)){
            return $this->parseInvalidParams();
        }

        $type = (int) $body->type;
        $allowed_types = Links::getTypes();

        if(!in_array($type, array_keys($allowed_types))){
            return $this->parseBadRequest();
        }

        $bad_request = $this->parseBadRequest($body, ['text']);

        if($type === 3 && is_wp_error($bad_request)){
            return $this->parseInvalidParams();
        } elseif(is_wp_error($this->parseBadRequest($body, ['link']))) {
            return $this->parseInvalidParams();
        }

        $unique_id = Utility::generateUniqueLinkID();
        $key = Utility::generateUniqueLinkKey();
        $password = isset($body->password) && !empty($body->password) ? $body->password : null;

        $params = [
            'title' => $body->title,
            'link' => $type !== 3 ? (isset($body->link) && !empty($body->link) ? $body->link : null) : 'Is Empty.',
            'type' => $type,
            'unique_id' => $unique_id,
            'key' => $key
        ];

        if (!empty($password)) {
            $params['password'] = $password;
        }

        $create = Links::insert($params);

        if($create){
            $data = [
                'unique_id' => $unique_id,
                'key' => $key,
                'safe_link' => Utility::getHashLink($unique_id)
            ];
            if($type === 2){
                $data['download_link'] = Utility::getDownloadLink($unique_id);
            }
            $this->result = [
                'message' => 'The has been successfully created.',
                'code' => 200,
                'data' => $data
            ];
            if($type === 3){
                LinkOptions::updateOption($unique_id, 'simple_text_content', $body->text);
            }
        } else {
            $this->result = [
                'message' => 'An error occurred, the request cannot be completed.',
                'code' => 400,
            ];
        }

        return self::parseResponse($this->result, $this->result['code']);
    }

    /**
     *
     * @param $request
     */
    public function handleGetResponse($request){
        $this->request = $request;
        $method = $this->request->get_method();
        $body = (object) $_GET;

        if($method !== 'GET'){
            return self::parseError('Method Not Allowed!', 405);
        }

        $many_request = $this->parseManyRequest();
        if(is_wp_error($many_request)){
            return $many_request;
        }

        $auth = $this->authorization($this->request);
        if(is_wp_error($auth)){
            return $auth;
        }

        $bad_request = $this->parseBadRequest($body, ['token']);
        if(is_wp_error($bad_request)){
            return $this->parseInvalidParams();
        }

        $token = $body->token;
        $result = (array) Links::get($token);

        if($result){
            $type = (int) $result['type'];
            $unique_id = $result['unique_id'];

            if($type === 3){
                unset($result['link']);
            }

            $result['safe_link'] = Utility::getHashLink($unique_id);

            if($type === 2){
                $result['download_link'] = Utility::getDownloadLink($unique_id);
            }

            $this->result = [
                'message' => 'The has been successfully.',
                'code' => 200,
                'data' => $result
            ];
        } else {
            $this->result = [
                'message' => 'Im Sorry, not found.',
                'code' => 404,
            ];
        }

        return self::parseResponse($this->result, $this->result['code']);
    }

    /**
     *
     * @param $request
     */
    public function handleRemoveResponse($request){
        $this->request = $request;
        $method = $this->request->get_method();
        $body = $this->getBody();

        if($method !== 'DELETE'){
            return self::parseError('Method Not Allowed!', 405);
        }

        $many_request = $this->parseManyRequest();
        if(is_wp_error($many_request)){
            return $many_request;
        }

        $auth = $this->authorization($this->request, 'write');
        if(is_wp_error($auth)){
            return $auth;
        }

        $bad_request = $this->parseBadRequest($body, ['token']);
        if(is_wp_error($bad_request)){
            return $this->parseInvalidParams();
        }

        $result = Links::get($body->token);
        if($result){
            $delete = Links::delete($result->unique_id);
            if($delete){
                $this->result = [
                    'message' => 'The has been successfully deleted.',
                    'code' => 200,
                ];
            } else {
                $this->result = [
                    'message' => 'An error occurred, the request cannot be completed.',
                    'code' => 400,
                ];
            }
        } else {
            $this->result = [
                'message' => 'Im Sorry, not found.',
                'code' => 404,
            ];
        }

        return self::parseResponse($this->result, $this->result['code']);
    }
}
