<?php
namespace WPSURL\App\Api;

use WPSURL\App\ApiKeys;
use WPSURL\App\Utility;

abstract class RestApi
{
    protected $request;
    protected $output;
    protected $auth_token;
    protected $result = [];

    public function __construct()
    {
        if(method_exists($this, 'registerEndpoint')){
            add_action('rest_api_init', [$this, 'registerEndpoint']);
        }
    }

    /**
     * @param $message
     * @param $status_code
     * @return void|mixed
     */
    public function parseError($message, $status_code){
        return new \WP_Error($status_code, $message, ['status' => $status_code]);
    }

    public function parseResponse($response, $status_code){
        return new \WP_REST_Response($response, $status_code);
    }

    /**
     * @param $body - request body
     * @param $params
     * @return mixed|void|\WP_Error|null
     */
    public function parseBadRequest($body = '', $params = []){
        if(empty($body) || !$this->paramsValidation($params, $body)){
            return self::parseError('Bad Request!', 400);
        }
    }

    /**
     * @param array $valid_params
     * @return mixed|\WP_Error|null
     */
    public function parseInvalidParams(array $valid_params = []){
        return self::parseError('Invalid Parameters!' . (!empty($valid_params) ? implode(',', $valid_params) : ''), 400);
    }

    /**
     * @param int $max_request
     * @param int $duration - in Seconds
     * @return mixed|void|\WP_Error|null
     */
    public function parseManyRequest(int $max_request = 5, int $duration = 120){
        $requestLimit = new RequestLimiter($max_request, $duration);

        if(!$requestLimit->checkRequest()){
            return self::parseError('There are too many requests, please try again in 2 minutes.', 429);
        }
    }

    public function getBody(){
        $json = Utility::isJson($this->request->get_body());
        return $json ? $json : false;
    }

    public function paramsValidation($params, $body){
        $return = true;
        if($params && $body){
            foreach ($params as $param){
                if(!isset($body->{$param}) || (isset($body->{$param}) && empty($body->{$param}))){
                    $return = false;
                }
            }
        }
        return !empty($body) ? $return : false;
    }

    public function authorization(\WP_REST_Request $request, $permission = 'read'){
        $headers = $request->get_headers();
        $auth_token = $headers['x_api_key'][0];
        $result = ApiKeys::getBySecret($auth_token);

        if (empty($auth_token) || !$result) {
            return $this->parseError(__('Not Authorized', WPSURL_TEXT_DOMAIN), 401);
        } elseif(!in_array($permission, explode('_', $result->permissions))){
            return $this->parseError(__('Permissions Denied', WPSURL_TEXT_DOMAIN), 401);
        }

        $update = ApiKeys::update($result->id, [
           'last_access' => current_time('mysql')
        ]);

        return true;
    }
}
