<?php
/**
 * Exit if accessed directly
 */
if (!defined('ABSPATH')) {
    exit;
}

if(class_exists('WPSURL_Plugin'))
    return false;

use WPSURL\App\Utility;
use WPSURL\App\Assets;

final class WPSURL_Plugin {
    public $hooks;

    private static $instance = null;

    /**
     * @return WPSURL_Plugin|null
     */
    public static function Instance(){
        if(!self::$instance){
            self::$instance = new self();
        }

        return self::$instance;
    }

    private function __construct()
    {
        $this->define_constants();
        $this->load_translations();
        add_action('plugins_loaded', [$this, 'include_files']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_front_assets'], 9999);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
    }

    private function define_constants(){
        define('WPSURL_IS_ADMIN', is_admin());
        define('WPSURL_VERSION', '1.0.0');
        define('WPSURL_TEXT_DOMAIN', 'wpsurl');
        define('WPSURL_PREFIX', 'wpsurl_');
        define('WPSURL_PLUGIN_BASENAME', plugin_basename(__FILE__));
        define('WPSURL_BASENAME', basename(dirname(__FILE__)));
        define('WPSURL_PATH', trailingslashit(plugin_dir_path(__FILE__)));
        define('WPSURL_APP_PATH', trailingslashit(WPSURL_PATH . 'app'));
        define('WPSURL_URL', trailingslashit(plugin_dir_url(__FILE__)));
        define('WPSURL_INC_PATH', trailingslashit(WPSURL_PATH . 'includes'));
        define('WPSURL_LIB_PATH', trailingslashit(WPSURL_INC_PATH . 'lib'));
        define('WPSURL_CLASS_PATH', trailingslashit(WPSURL_INC_PATH . 'classes'));
        define('WPSURL_ADMIN_PATH', trailingslashit(WPSURL_PATH . 'admin'));
        define('WPSURL_SETTINGS_PATH', trailingslashit(WPSURL_ADMIN_PATH . 'pages/settings'));
        define('WPSURL_CSS_URL', trailingslashit(WPSURL_URL . 'assets/css'));
        define('WPSURL_ELEMENTOR_CSS_URL', trailingslashit(WPSURL_CSS_URL . 'elementor'));
        define('WPSURL_JS_URL', trailingslashit(WPSURL_URL . 'assets/js'));
        define('WPSURL_IMG_URL', trailingslashit(WPSURL_URL . 'assets/images'));
        define('WPSURL_VIEW_PATH', trailingslashit(WPSURL_PATH . 'view'));
        define('WPSURL_VIEW_INC', trailingslashit(WPSURL_VIEW_PATH . 'includes'));
        define('WPSURL_VIEW_PAGES', trailingslashit(WPSURL_VIEW_PATH . 'pages'));
        define('WPSURL_VIEW_ADMIN_PATH', trailingslashit(WPSURL_VIEW_PATH . 'admin'));
    }

    /**
     * Include init files
     *
     * @return void
     */
    public function include_files(){
        require_once WPSURL_PATH . 'vendor/autoload.php';
        require_once WPSURL_LIB_PATH . 'jalali.php';
        // Auto-load classes on demand
        if (function_exists("__autoload")) {
            spl_autoload_register("__autoload");
        }
        spl_autoload_register([$this, 'autoloader']);

        ob_start();

        // Globals Var
        global $loops, $wpgallery, $alert, $geoplugin;
        // global flash message class
        $alert = \WPSURL\App\Alert::Instance();
        // global geo plugin class
        $geoplugin = \WPSURL\App\GeoPlugin::Instance();

        // global class downloader
        if (WPSURL_IS_ADMIN) {
            // global wp gallery class
            $wpgallery = \WPSURL\App\WPGallery::Instance();
            // global loops class
            $loops = \WPSURL\App\Links::Instance();
        }

        $hooks = \WPSURL\App\Hooks::Instance();

        // --------------------------------------------------------------------------------------------------- Start RTL License
        /*$rtlLicenseClassName  = 'RTL_License_f5c64599271c20fa';
        $rtlLicenseFilePath   = WPSURL_INC_PATH . $rtlLicenseClassName . '.php';
        $rtlLicenseFileHash   = @sha1_file($rtlLicenseFilePath);

        if ( $rtlLicenseFileHash === 'd2c3e14a2ea3f1c0d1dd2661a8328ea54727a332' && file_exists($rtlLicenseFilePath) ) {
            require_once $rtlLicenseFilePath;

            if ( class_exists($rtlLicenseClassName) && method_exists($rtlLicenseClassName, 'isActive') ) {
                $rtlLicenseClass = new $rtlLicenseClassName();

                if ( $rtlLicenseClass->{'isActive'}() === true ) {
                    $hooks = \WPSURL\App\Hooks::Instance();
                }
            }
        }*/
        // ----------------------------------------------------------------------------------------------------- End RTL License
    }

    /**
     * Enqueue front scripts
     *
     * @return void
     */
    public function enqueue_front_assets(){
        if (Utility::issetQueryVars()) {
            if (!is_admin()) {
                wp_deregister_script('jquery');
                Assets::enqueueScript('jquery', WPSURL_JS_URL . 'jquery.min.js', false);
            }
            Assets::enqueueStyle('wpsurl.grid.css', WPSURL_CSS_URL . 'grid.min.css', null, WPSURL_VERSION);
            Assets::enqueueStyle('wpsurl.icons.css', WPSURL_CSS_URL . 'icons.css', null, WPSURL_VERSION);
            Assets::enqueueStyle('wpsurl.izitoast.css', WPSURL_CSS_URL . 'iziToast.min.css', null, WPSURL_VERSION);
            Assets::enqueueStyle('wpsurl.hystmodal.css', WPSURL_CSS_URL . 'hystmodal.min.css', null, WPSURL_VERSION);
            Assets::enqueueStyle('wpsurl.bulma.css', WPSURL_CSS_URL . 'bulma.min.css', null, WPSURL_VERSION);
            Assets::enqueueStyle('wpsurl.ui.css', WPSURL_CSS_URL . 'ui.css', null, WPSURL_VERSION);
            if(!is_rtl()){
                Assets::enqueueStyle('wpsurl.ui.ltr.css', WPSURL_CSS_URL . 'ui-ltr.css', null, WPSURL_VERSION);
            }
            Assets::enqueueStyle('wpsurl.iframe.css', WPSURL_CSS_URL . 'iframe.css', null, WPSURL_VERSION);
            if(!is_rtl()){
                Assets::enqueueStyle('wpsurl.iframe.ltr.css', WPSURL_CSS_URL . 'iframe-ltr.css', null, WPSURL_VERSION);
            }
            Assets::enqueueStyle('wpsurl.assets.css', WPSURL_CSS_URL . 'assets.css', null, WPSURL_VERSION);
            if(!is_rtl()){
                Assets::enqueueStyle('wpsurl.assets.ltr.css', WPSURL_CSS_URL . 'assets-ltr.css', null, WPSURL_VERSION);
            }
            Assets::enqueueStyle('wpsurl.responsive.css', WPSURL_CSS_URL . 'responsive.css', null, WPSURL_VERSION);

            Assets::enqueueScript('wpsurl.izitoast.js', WPSURL_JS_URL . 'iziToast.min.js', null, WPSURL_VERSION, true);
            Assets::enqueueScript('wpsurl.hystmodal.js', WPSURL_JS_URL . 'hystmodal.min.js', null, WPSURL_VERSION, true);
            Assets::enqueueScript('wpsurl.persian-tools.js', WPSURL_JS_URL . 'persian-tools.umd.js', null, WPSURL_VERSION, true);
            Assets::enqueueScript('wpsurl.md5.js', WPSURL_JS_URL . 'md5.min.js', null, WPSURL_VERSION, true);
            Assets::enqueueScript('wpsurl.assets.js', WPSURL_JS_URL . 'assets.js', array('jquery'), WPSURL_VERSION, true);

            if (!\WPSURL\App\Settings::getOption( 'view_timer_status' )){
                Assets::enqueueScript('wpsurl.countdown.js', WPSURL_JS_URL . 'countdown.js', null, WPSURL_VERSION, true);
                wp_localize_script(Assets::getHandleName('wpsurl.countdown.js'), 'wpsurl_data', Assets::getLocalizeData());
            }
            Assets::enqueueScript('wpsurl.ui.js', WPSURL_JS_URL . 'ui.js', array('jquery'), WPSURL_VERSION, true);
            wp_localize_script(Assets::getHandleName('wpsurl.ui.js'), 'wpsurl_data', Assets::getLocalizeData());
            Assets::enqueueScript('wpsurl.ajax.js', WPSURL_JS_URL . 'ajax.js', array('jquery'), WPSURL_VERSION, true);
            wp_localize_script(Assets::getHandleName('wpsurl.ajax.js'), 'wpsurl_data', Assets::getLocalizeData());
        } else {
            Assets::enqueueStyle('wpsurl.site.css', WPSURL_CSS_URL . 'site.css', null, WPSURL_VERSION);
        }
    }

    /**
     * Enqueue admin scripts
     *
     * @return void
     */
    public function enqueue_admin_assets(){
        Assets::enqueueStyle('wpsurl.admin.css', WPSURL_CSS_URL . 'admin.css', null, WPSURL_VERSION);
        if(!is_rtl()){
            Assets::enqueueStyle('wpsurl.admin.ltr.css', WPSURL_CSS_URL . 'admin-ltr.css', null, WPSURL_VERSION);
        }
        Assets::enqueueStyle('wpsurl.settings.css', WPSURL_CSS_URL . 'settings.css', null, WPSURL_VERSION);
        Assets::enqueueStyle('wpsurl.icons.css', WPSURL_CSS_URL . 'icons.css', null, WPSURL_VERSION);
        Assets::enqueueStyle('wpsurl.selectize.css', WPSURL_CSS_URL . 'selectize.default.css', null, WPSURL_VERSION);
        Assets::enqueueStyle('wpsurl.izitoast.css', WPSURL_CSS_URL . 'iziToast.min.css', null, WPSURL_VERSION);
        Assets::enqueueStyle('wpsurl.assets.css', WPSURL_CSS_URL . 'assets.css', null, WPSURL_VERSION);
        Assets::enqueueStyle('wpsurl.responsive.css', WPSURL_CSS_URL . 'responsive.css', null, WPSURL_VERSION);
        Assets::enqueueScript('wpsurl.selectize.js', WPSURL_JS_URL . 'selectize.min.js', array('jquery',), WPSURL_VERSION);
        Assets::enqueueScript('wpsurl.izitoast.js', WPSURL_JS_URL . 'iziToast.min.js', null, WPSURL_VERSION, true);
        Assets::enqueueScript('wpsurl.persian-tools.js', WPSURL_JS_URL . 'persian-tools.umd.js', null, WPSURL_VERSION, true);
        Assets::enqueueScript('wpsurl.assets.js', WPSURL_JS_URL . 'assets.js', array('jquery'), WPSURL_VERSION, true);
        Assets::enqueueScript('wpsurl.admin.js', WPSURL_JS_URL . 'admin.js', array('jquery',), WPSURL_VERSION);
        wp_localize_script(Assets::getHandleName('wpsurl.admin.js'), 'wpsurl_data', Assets::getLocalizeData());
        if (isset($_GET['page']) && $_GET['page'] == 'wp-safe-url') {
            Assets::enqueueScript('wp-theme-plugin-editor');
            Assets::enqueueStyle('wp-codemirror');
            wp_enqueue_media();
        }
    }

    /**
     * Class autoloader
     *
     * @param string $class
     * @return void
     */
    public function autoloader($class){
        if(strpos($class, 'WPSURL') !== false){
            $class = str_replace(['WPSURL\\', 'WPSURL', '\\'], ['', '', '/'], $class);
            $class_arr = explode('/', $class);
            $file_name = $class_arr[array_key_last($class_arr)] . '.php';
            unset($class_arr[array_key_last($class_arr)]);
            $file_path = WPSURL_PATH . strtolower(implode('/', $class_arr)) . '/' . $file_name;
           
            if(file_exists($file_path) && is_readable($file_path)){
                include_once($file_path);
            }
        }
    }

    /**
     * 
     * 
     * Load text domain
     * 
     */
    public function load_translations(){
        $locale = apply_filters('plugin_locale', get_locale(), WPSURL_TEXT_DOMAIN);
        $wp_core_lang = trailingslashit(WP_LANG_DIR) . WPSURL_TEXT_DOMAIN . '/' . WPSURL_TEXT_DOMAIN . '-' . $locale . '.mo';
        if (file_exists($wp_core_lang)) {
            load_textdomain(WPSURL_TEXT_DOMAIN, $wp_core_lang);
        }
        load_plugin_textdomain(WPSURL_TEXT_DOMAIN, false, WPSURL_BASENAME . '/languages/');
    }
}